<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

function razer_get_order_meta( $order_id, $key, $single = true ) {
    if (
        class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) &&
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()
    ) {
        $order = wc_get_order( $order_id );
        if ( $order ) {
            return $order->get_meta( $key, $single );
        }
        return $single ? '' : array();
    }
    return get_post_meta( $order_id, $key, $single );
}

function razer_update_order_meta( $order_id, $key, $value ) {
    if (
        class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) &&
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()
    ) {
        $order = wc_get_order( $order_id );
        if ( $order ) {
            $order->update_meta_data( $key, $value );
            $order->save();
            return;
        }
    }
    update_post_meta( $order_id, $key, $value );
}

function razer_delete_order_meta( $order_id, $key ) {
    if (
        class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) &&
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()
    ) {
        $order = wc_get_order( $order_id );
        if ( $order ) {
            $order->delete_meta_data( $key );
            $order->save();
            return;
        }
    }
    delete_post_meta( $order_id, $key );
}

function razer_add_order_meta( $order_id, $key, $value, $unique = false ) {
    if (
        class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) &&
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()
    ) {
        $order = wc_get_order( $order_id );
        if ( $order ) {
            if ( $unique ) {
                $existing = $order->get_meta( $key, true );
                if ( ! empty( $existing ) ) {
                    return;
                }
            }
            $order->add_meta_data( $key, $value, $unique );
            $order->save();
            return;
        }
    }
    add_post_meta( $order_id, $key, $value, $unique );
}

function razer_send_order_to_server( $order_id ) {
    razer_log( "=== بدء معالجة الطلب #$order_id ===" );

    $order = wc_get_order( $order_id );
    if ( ! $order ) {
        razer_log( 'Order not found for ID: ' . $order_id );
        return;
    }

    $already_sent = razer_get_order_meta( $order_id, '_razer_order_sent' );
    if ( 'yes' === $already_sent ) {
        razer_log( "Order #$order_id has already been sent" );
        return;
    }

    $remote_ids = array();

    foreach ( $order->get_items() as $item_id => $item ) {
        $product_id   = $item->get_product_id();
        $variation_id = $item->get_variation_id();
        $check_id     = $variation_id ? $variation_id : $product_id;

        $enabled         = get_post_meta( $check_id, '_razer_integration_enabled', true );
        $product_mapping = get_post_meta( $check_id, '_razer_product_mapping', true );
        $player_id       = $item->get_meta( '_razer_player_id' );

        razer_log(
            sprintf(
                'Checking item #%d (product #%d): enabled=%s, mapping=%s, player_id=%s',
                $item_id,
                $check_id,
                $enabled,
                $product_mapping,
                $player_id
            )
        );

        if ( 'yes' !== $enabled || empty( $product_mapping ) ) {
            razer_log( "Skipping item #$item_id (not enabled or no mapping)." );
            continue;
        }

        $enable_player_id = get_post_meta( $check_id, '_razer_enable_player_id', true );
        if ( 'yes' === $enable_player_id && empty( $player_id ) ) {
            $order->add_order_note(
                sprintf(
                    '❌ Razer Gold: معرف اللاعب مطلوب للمنتج %s لكنه مفقود. لم يتم إرسال الطلب.',
                    $item->get_name()
                )
            );
            razer_log( '❌ Player ID required but missing for item #' . $item_id );
            continue;
        }

        $qty          = (int) $item->get_quantity();
        $external_ref = $order_id . '-' . $item_id;

        $body = array(
            'player_id'    => $player_id,
            'product_name' => $product_mapping,
            'quantity'     => $qty,
            'external_ref' => $external_ref,
        );

        razer_log( '🚀 Sending request to Razer Gold Bot (POST /api/order)' );

        $result = razer_api_request(
            'POST',
            'api/order',
            array( 'body' => $body )
        );

        if ( is_wp_error( $result ) ) {
            $order->add_order_note(
                sprintf(
                    '❌ Razer Gold Bot فشل الاتصال: %s',
                    $result->get_error_message()
                )
            );
            razer_log( '❌ WP_Error: ' . $result->get_error_message() );
            continue;
        }

        if ( ! empty( $result['success'] ) && ! empty( $result['order_id'] ) ) {
            $remote_order_id = (string) $result['order_id'];
            $remote_ids[]    = $remote_order_id;

            $item->add_meta_data( 'رقم طلب البوت', $remote_order_id, true );
            $item->add_meta_data( '_razer_remote_order_id', $remote_order_id, true );
            $item->save();

            $order->add_order_note(
                sprintf(
                    '✅ تم إنشاء طلب البوت: #%s للمنتج %s',
                    $remote_order_id,
                    $item->get_name()
                )
            );

            razer_log( '✅ Remote order created: ' . $remote_order_id . ' for WC order #' . $order_id );

            razer_add_remote_id_to_order( $order_id, $remote_order_id );
        } else {
            $error_msg = isset( $result['message'] ) ? $result['message'] : 'خطأ غير معروف';
            $order->add_order_note(
                sprintf( '❌ Razer Gold Bot خطأ: %s', $error_msg )
            );
            razer_log( '❌ Error: ' . $error_msg );
        }
    }

    if ( ! empty( $remote_ids ) ) {
        razer_update_order_meta( $order_id, '_razer_remote_ids', implode( ',', $remote_ids ) );
        razer_start_order_polling( $order_id );
        razer_update_order_meta( $order_id, '_razer_order_sent', 'yes' );
        razer_log( '✅ Saved remote order IDs for order #' . $order_id . ': ' . implode( ',', $remote_ids ) );
    }

    razer_log( "=== انتهت معالجة الطلب #$order_id ===" );
}
add_action( 'woocommerce_order_status_processing', 'razer_send_order_to_server', 10, 1 );

function razer_add_remote_id_to_order( $order_id, $remote_id ) {
    $order_id  = (int) $order_id;
    $remote_id = sanitize_text_field( (string) $remote_id );

    if ( empty( $remote_id ) ) {
        return;
    }

    $existing = razer_get_order_meta( $order_id, '_razer_remote_id', false );
    if ( is_array( $existing ) && in_array( $remote_id, $existing, true ) ) {
        return;
    }

    razer_add_order_meta( $order_id, '_razer_remote_id', $remote_id, false );
}

function razer_start_order_polling( $order_id ) {
    $order_id = (int) $order_id;

    if ( razer_get_order_meta( $order_id, '_razer_status_final' ) ) {
        return;
    }

    $remote_ids = razer_get_order_meta( $order_id, '_razer_remote_ids' );
    if ( empty( $remote_ids ) ) {
        return;
    }

    $existing_next = razer_get_order_meta( $order_id, '_razer_next_check_at' );
    if ( ! empty( $existing_next ) ) {
        return;
    }

    razer_update_order_meta( $order_id, '_razer_poll_attempt', 0 );
    razer_update_order_meta( $order_id, '_razer_next_check_at', time() + 60 );
    razer_delete_order_meta( $order_id, '_razer_status_final' );

    razer_log( "Initialized polling for order #$order_id (next in 60s)" );
}

function razer_get_poll_delays() {
    return apply_filters( 'razer_poll_delays', array( 60, 120, 300, 600, 1200, 1800, 3600 ) );
}

function razer_get_poll_batch_size() {
    return (int) apply_filters( 'razer_poll_batch_size', 25 );
}

function razer_fetch_remote_status( $remote_id ) {
    $remote_id = sanitize_text_field( (string) $remote_id );

    $result = razer_api_request( 'GET', 'api/order/' . rawurlencode( $remote_id ) );

    if ( is_wp_error( $result ) ) {
        return $result;
    }

    $status = isset( $result['status'] ) ? strtolower( (string) $result['status'] ) : '';

    return array(
        'remote_status'  => $status,
        'pin_code'       => isset( $result['pin_code'] ) ? $result['pin_code'] : '',
        'transaction_id' => isset( $result['transaction_id'] ) ? $result['transaction_id'] : '',
        'error_message'  => isset( $result['error_message'] ) ? $result['error_message'] : '',
        'raw'            => $result,
    );
}

function razer_poll_pending_orders_handler() {
    if ( get_transient( 'razer_poll_lock' ) ) {
        return;
    }
    set_transient( 'razer_poll_lock', 1, 55 );

    global $wpdb;

    $now     = time();
    $limit   = razer_get_poll_batch_size();
    $delays  = razer_get_poll_delays();
    $max_try = count( $delays );

    $statuses = array( 'wc-processing', 'wc-on-hold', 'wc-pending' );
    $status_placeholders = implode( ',', array_fill( 0, count( $statuses ), '%s' ) );

    $order_ids = array();

    if (
        class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) &&
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()
    ) {
        $orders_table = $wpdb->prefix . 'wc_orders';
        $meta_table   = $wpdb->prefix . 'wc_orders_meta';

        $sql = "
            SELECT o.id
            FROM {$orders_table} o
            INNER JOIN {$meta_table} m_next
                ON m_next.order_id = o.id
               AND m_next.meta_key = '_razer_next_check_at'
               AND CAST(m_next.meta_value AS UNSIGNED) <= %d
            INNER JOIN {$meta_table} m_remote
                ON m_remote.order_id = o.id
               AND m_remote.meta_key = '_razer_remote_ids'
            LEFT JOIN {$meta_table} m_final
                ON m_final.order_id = o.id
               AND m_final.meta_key = '_razer_status_final'
            WHERE o.status IN ({$status_placeholders})
              AND m_final.order_id IS NULL
            ORDER BY o.date_created_gmt ASC
            LIMIT %d
        ";

        $params    = array_merge( array( $now ), $statuses, array( $limit ) );
        $order_ids = $wpdb->get_col( $wpdb->prepare( $sql, ...$params ) );
    } else {
        $posts_table = $wpdb->posts;
        $meta_table  = $wpdb->postmeta;

        $sql = "
            SELECT p.ID
            FROM {$posts_table} p
            INNER JOIN {$meta_table} m_next
                ON m_next.post_id = p.ID
               AND m_next.meta_key = '_razer_next_check_at'
               AND CAST(m_next.meta_value AS UNSIGNED) <= %d
            INNER JOIN {$meta_table} m_remote
                ON m_remote.post_id = p.ID
               AND m_remote.meta_key = '_razer_remote_ids'
            LEFT JOIN {$meta_table} m_final
                ON m_final.post_id = p.ID
               AND m_final.meta_key = '_razer_status_final'
            WHERE p.post_type = 'shop_order'
              AND p.post_status IN ({$status_placeholders})
              AND m_final.post_id IS NULL
            ORDER BY p.post_date ASC
            LIMIT %d
        ";

        $params    = array_merge( array( $now ), $statuses, array( $limit ) );
        $order_ids = $wpdb->get_col( $wpdb->prepare( $sql, ...$params ) );
    }

    if ( empty( $order_ids ) ) {
        delete_transient( 'razer_poll_lock' );
        return;
    }

    foreach ( $order_ids as $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            continue;
        }

        if ( razer_get_order_meta( $order_id, '_razer_status_final' ) ) {
            razer_delete_order_meta( $order_id, '_razer_next_check_at' );
            continue;
        }

        $remote_ids_str = (string) razer_get_order_meta( $order_id, '_razer_remote_ids' );
        $remote_ids_str = trim( $remote_ids_str );

        if ( empty( $remote_ids_str ) ) {
            razer_delete_order_meta( $order_id, '_razer_next_check_at' );
            continue;
        }

        $remote_ids = array_values(
            array_filter(
                array_map( 'trim', explode( ',', $remote_ids_str ) )
            )
        );

        $attempt = (int) razer_get_order_meta( $order_id, '_razer_poll_attempt' );
        $attempt = max( 0, $attempt );

        razer_log( "Polling order #$order_id attempt=" . ( $attempt + 1 ) . " remote_ids=" . $remote_ids_str );

        $any_failed    = false;
        $all_completed = true;
        $had_error     = false;
        $notes         = array();

        foreach ( $remote_ids as $rid ) {
            $status_res = razer_fetch_remote_status( $rid );

            if ( is_wp_error( $status_res ) ) {
                $had_error = true;
                razer_log( 'Polling error for remote_id=' . $rid . ': ' . $status_res->get_error_message() );
                continue;
            }

            $remote_status = $status_res['remote_status'];

            if ( 'completed' === $remote_status ) {
                $note_parts = array( '✅ طلب البوت #' . $rid . ' مكتمل' );
                if ( ! empty( $status_res['pin_code'] ) ) {
                    $note_parts[] = 'رمز PIN: ' . $status_res['pin_code'];
                }
                if ( ! empty( $status_res['transaction_id'] ) ) {
                    $note_parts[] = 'رقم العملية: ' . $status_res['transaction_id'];
                }
                $notes[] = implode( ' | ', $note_parts );
                continue;
            }

            if ( 'failed' === $remote_status ) {
                $any_failed    = true;
                $all_completed = false;
                $error_msg     = ! empty( $status_res['error_message'] ) ? $status_res['error_message'] : 'خطأ غير محدد';
                $notes[]       = '❌ طلب البوت #' . $rid . ' فشل: ' . $error_msg;
                break;
            }

            $all_completed = false;
        }

        $final          = false;
        $current_status = $order->get_status();
        $allowed_from   = array( 'pending', 'processing', 'on-hold' );

        if ( in_array( $current_status, $allowed_from, true ) ) {
            if ( $any_failed ) {
                foreach ( $notes as $note ) {
                    $order->add_order_note( $note );
                }
                $order->update_status( 'failed', '❌ تم رفض/فشل الطلب من Razer Gold Bot' );
                $final = true;
            } elseif ( $all_completed && ! $had_error ) {
                foreach ( $notes as $note ) {
                    $order->add_order_note( $note );
                }
                $order->update_status( 'completed', '✅ تم إكمال الطلب من Razer Gold Bot' );
                $final = true;
            }
        }

        if ( $final ) {
            razer_update_order_meta( $order_id, '_razer_status_final', 1 );
            razer_delete_order_meta( $order_id, '_razer_next_check_at' );
            razer_log( 'Order #' . $order_id . ' marked ' . $order->get_status() . ' (from polling).' );
            continue;
        }

        $attempt++;

        if ( $attempt >= $max_try ) {
            razer_delete_order_meta( $order_id, '_razer_next_check_at' );
            $order->add_order_note(
                '⌛ Razer Gold Bot: لم يصل رد نهائي بعد ساعة. سيتم ترك الطلب قيد التنفيذ، ويمكن استكماله عبر الويبهوك لاحقاً.'
            );
            razer_log( 'Order #' . $order_id . ' polling stopped after max attempts; order left as-is.' );
            continue;
        }

        razer_update_order_meta( $order_id, '_razer_poll_attempt', $attempt );
        razer_update_order_meta( $order_id, '_razer_next_check_at', $now + (int) $delays[ $attempt ] );

        if ( in_array( $attempt, array( 2, 4, 6 ), true ) ) {
            $order->add_order_note(
                sprintf(
                    '⌛ Razer Gold Bot: ما زال الطلب قيد المعالجة — محاولة %d/%d',
                    $attempt + 1,
                    $max_try
                )
            );
        }
    }

    delete_transient( 'razer_poll_lock' );
}
add_action( 'razer_poll_pending_orders', 'razer_poll_pending_orders_handler' );
