<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

function razer_register_webhook_route() {
    register_rest_route( 'razer-gold/v1', '/webhook', array(
        'methods'             => 'POST',
        'callback'            => 'razer_handle_webhook',
        'permission_callback' => '__return_true',
    ) );
}
add_action( 'rest_api_init', 'razer_register_webhook_route' );

function razer_handle_webhook( WP_REST_Request $request ) {
    $webhook_secret = get_option( 'razer_webhook_secret', '' );

    if ( ! empty( $webhook_secret ) ) {
        $provided_secret = $request->get_header( 'X-Webhook-Secret' );
        if ( $provided_secret !== $webhook_secret ) {
            razer_log( '❌ Webhook: مفتاح التحقق غير صحيح' );
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'مفتاح التحقق غير صحيح',
            ), 403 );
        }
    }

    $body = $request->get_json_params();

    if ( empty( $body ) ) {
        return new WP_REST_Response( array(
            'success' => false,
            'message' => 'لا توجد بيانات',
        ), 400 );
    }

    $remote_order_id = isset( $body['order_id'] ) ? sanitize_text_field( (string) $body['order_id'] ) : '';
    $status          = isset( $body['status'] ) ? sanitize_text_field( strtolower( $body['status'] ) ) : '';
    $pin_code        = isset( $body['pin_code'] ) ? sanitize_text_field( $body['pin_code'] ) : '';
    $transaction_id  = isset( $body['transaction_id'] ) ? sanitize_text_field( $body['transaction_id'] ) : '';
    $error_message   = isset( $body['error_message'] ) ? sanitize_text_field( $body['error_message'] ) : '';

    razer_log( "Webhook received: order_id=$remote_order_id, status=$status" );

    if ( empty( $remote_order_id ) || empty( $status ) ) {
        return new WP_REST_Response( array(
            'success' => false,
            'message' => 'order_id و status مطلوبان',
        ), 400 );
    }

    $wc_order_id = razer_find_order_by_remote_id( $remote_order_id );

    if ( ! $wc_order_id ) {
        razer_log( '❌ Webhook: لم يتم العثور على طلب WooCommerce لـ remote_id=' . $remote_order_id );
        return new WP_REST_Response( array(
            'success' => false,
            'message' => 'لم يتم العثور على الطلب',
        ), 404 );
    }

    $order = wc_get_order( $wc_order_id );
    if ( ! $order ) {
        return new WP_REST_Response( array(
            'success' => false,
            'message' => 'خطأ في تحميل الطلب',
        ), 500 );
    }

    $current_status = $order->get_status();
    $allowed_from   = array( 'pending', 'processing', 'on-hold' );

    if ( ! in_array( $current_status, $allowed_from, true ) ) {
        razer_log( "Webhook: Order #$wc_order_id already in status '$current_status', skipping." );
        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'الطلب في حالة نهائية بالفعل',
        ), 200 );
    }

    if ( 'completed' === $status ) {
        $note_parts = array( '✅ Webhook: طلب البوت #' . $remote_order_id . ' مكتمل' );
        if ( ! empty( $pin_code ) ) {
            $note_parts[] = 'رمز PIN: ' . $pin_code;
        }
        if ( ! empty( $transaction_id ) ) {
            $note_parts[] = 'رقم العملية: ' . $transaction_id;
        }
        $order->add_order_note( implode( ' | ', $note_parts ) );
        $order->update_status( 'completed', '✅ تم إكمال الطلب عبر Webhook' );

        razer_update_order_meta( $wc_order_id, '_razer_status_final', 1 );
        razer_delete_order_meta( $wc_order_id, '_razer_next_check_at' );

        razer_log( "Webhook: Order #$wc_order_id marked completed." );
    } elseif ( 'failed' === $status ) {
        $note = '❌ Webhook: طلب البوت #' . $remote_order_id . ' فشل';
        if ( ! empty( $error_message ) ) {
            $note .= ': ' . $error_message;
        }
        $order->add_order_note( $note );
        $order->update_status( 'failed', '❌ فشل الطلب عبر Webhook' );

        razer_update_order_meta( $wc_order_id, '_razer_status_final', 1 );
        razer_delete_order_meta( $wc_order_id, '_razer_next_check_at' );

        razer_log( "Webhook: Order #$wc_order_id marked failed." );
    } else {
        $order->add_order_note(
            sprintf( 'ℹ️ Webhook: حالة طلب البوت #%s: %s', $remote_order_id, $status )
        );
        razer_log( "Webhook: Order #$wc_order_id status update: $status" );
    }

    return new WP_REST_Response( array(
        'success' => true,
        'message' => 'تم معالجة الويبهوك بنجاح',
    ), 200 );
}

function razer_find_order_by_remote_id( $remote_id ) {
    global $wpdb;

    $remote_id = sanitize_text_field( (string) $remote_id );

    if (
        class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) &&
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()
    ) {
        $meta_table = $wpdb->prefix . 'wc_orders_meta';
        $order_id   = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT order_id FROM {$meta_table} WHERE meta_key = '_razer_remote_id' AND meta_value = %s LIMIT 1",
                $remote_id
            )
        );
    } else {
        $order_id = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_razer_remote_id' AND meta_value = %s LIMIT 1",
                $remote_id
            )
        );
    }

    return $order_id ? (int) $order_id : false;
}
